// 
//  Ethernet based relay controller
//
//  A stand alond relay controller board, implementing a simple state manchine providing
//  autonomous operation.  Able to be programmed via an IP network.
//
//  This software uses the etherShield library and associated setup code.
//
//
// (C) 2010 Doug Jackson - Licensed under the TAPR Open Hardware License (www.tapr.org/NCL)
//
// Extensive use has been made of the NuElectronics Ethershield example code
//
////////////////////////////////////////////////
//
//  V1.0 20100903 - DRJ - First Production Version (YAY! - that was fun!)
//
////////////////////////////////////////////////
//  Available Web pages:
//  /          Main Page - Displays a menu of options and status information
//  /?cmd=1    Program page - Displays the current program settings for state 0 - 6
//  /?cmd=2    Program page - Displays the current program settings for state 7 - 14
//  /?cmd=3    Program page - Displays the current program settings for state 15 - 20
//  /?cmd=4    Manual settings page - Make manual setting changes to relays and allow timer to be turned on or off
//  /?cmd=4?rly=1?tmr=0 - format for returning manual settings toggles relay1 state and turns off timer
//  /?cmd=5?st=1   Edit Program page  - Allows changes to be made to the state settings
//  /?cmd=5?st=1?wait=2?nextt=2?o1=0?o2=1?o3=0?o4=0?o5=1?o6=1 - format for returning state changes
//
//

#include "etherShield.h"
#include <stdio.h>
#include <avr/pgmspace.h>
#include <string.h>
#include <EEPROM.h>

// please modify the following two lines. mac and ip have to be unique
// in your local area network. You can not have the same numbers in
// two devices:
static uint8_t mymac[6] = {0x54,0x55,0x58,0x10,0x00,0x25}; 
static uint8_t myip[4] = {192,168,1,2};
static char  baseurl[]="http://192.168.1.2/";
static uint16_t mywwwport =80; // listen port for tcp/www (max range 1-254)

prog_char header[] PROGMEM ="HTTP/1.0 200 OK\r\nContent-Type: text/html\r\n\r\n";
prog_char command_line[] PROGMEM ="<h3><a href=\"/?cmd=1\">Program 0-6</a> &nbsp;<a href=\"/?cmd=2\">Program 7-13</a> &nbsp;<a href=\"/?cmd=3\">Program 14-19</a>&nbsp;<a href=\"/?cmd=4\">Manual</a>&nbsp;</h3>";
prog_char djcopyright[] PROGMEM = "";  //<p>V1.0 - (C)2010 Doug Jackson";

// create a string table pointing to our PROGMEM strings (Anything to save RAM!!!)
PROGMEM const char *string_table[]=
{ header,
  command_line,
  djcopyright
};

// Ethernet buffers
// These are very problematic - Have to make sure that each web page returned is
// smaller than this buffer size.  Increasing the buffer will impact on stack space
// and everything will stop!! - You have been warned!
#define BUFFER_SIZE 980
static uint8_t buf[BUFFER_SIZE+1];


// A structure to hold our state table
//
typedef struct program_internal
{
  uint8_t output;
  uint16_t wait;
  uint8_t nextstate;
} statetable;

#define NUM_STATES 20
statetable states[NUM_STATES];


static uint16_t wait_timer;   // The countdown timer for the current state
static uint8_t current_state; // The storage for our current state
uint16_t r;      // a return value register used by various processes - defined as a global so each routine
                 // wasn't creating one!

char relay_string[6];         // a character array used to store various relay strings

// a temporary string buffer
#define LINE_BUFFER_SIZE 200
char line_buffer[LINE_BUFFER_SIZE];


EtherShield es=EtherShield();


// Hardware definitions to match the new PCB I designed.
// The actual relays are connected to the following pins
#define RELAY1 9
#define RELAY2 5
#define RELAY3 6
#define RELAY4 7
#define RELAY5 8
#define RELAY6 4
uint8_t RELAY[6]={RELAY1, RELAY2, RELAY3, RELAY4, RELAY5, RELAY6};

static byte relay_status;  // A variable to store which relays are on at the moment
byte run_status;           // holds the status of our timer - 1 = running 0 = on hold for manual operation



void setup()
{
  uint8_t i;
  
  Serial.begin(9600);
  //Serial.println("Ethernet Relay Controller v1.0");
  Serial.println("Reset relays");
  for (i=0; i<6; i++)  pinMode(RELAY[i], OUTPUT);  // start with the relays turned off
  relay_status=0;
  
  Serial.println("Init Enc28J60");
  //initialize enc28j60
  es.ES_enc28j60Init(mymac);
  es.ES_enc28j60clkout(2); // change clkout from 6.25MHz to 12.5MHz
  delay(10);
        
  // Magjack leds configuration, see enc28j60 datasheet, page 11 
  // LEDA=greed LEDB=yellow
  // 0x880 is PHLCON LEDB=on, LEDA=on
  // enc28j60PhyWrite(PHLCON,0b0000 1000 1000 00 00);
  es.ES_enc28j60PhyWrite(PHLCON,0x880);
  delay(500);
  //
  // 0x990 is PHLCON LEDB=off, LEDA=off
  // enc28j60PhyWrite(PHLCON,0b0000 1001 1001 00 00);
  es.ES_enc28j60PhyWrite(PHLCON,0x990);
  delay(500);
  //
  // 0x880 is PHLCON LEDB=on, LEDA=on
  // enc28j60PhyWrite(PHLCON,0b0000 1000 1000 00 00);
  es.ES_enc28j60PhyWrite(PHLCON,0x880);
  delay(500);
  //
  // 0x990 is PHLCON LEDB=off, LEDA=off
  // enc28j60PhyWrite(PHLCON,0b0000 1001 1001 00 00);
  es.ES_enc28j60PhyWrite(PHLCON,0x990);
  delay(500);
  //
  // 0x476 is PHLCON LEDA=links status, LEDB=receive/transmit
  // enc28j60PhyWrite(PHLCON,0b0000 0100 0111 01 10);
  es.ES_enc28j60PhyWrite(PHLCON,0x476);
  delay(100);
        
  //init the ethernet/ip layer:
  es.ES_init_ip_arp_udp_tcp(mymac,myip,80);
    


  
  if(EEPROM.read(0) != 0x5a)
  {
    Serial.println("EEPROM unitialized");

    EEPROM.write(0, 0x5a);    // Signature
    EEPROM.write(1, 0x01);    // 1 = Running on normal timer..   0 = halted for manual use
 
 #define NUM_DEFAULT_STATES 8
    
    states[0].output = B00101010;  // what relays are on?
    states[0].wait = 1;            // delay time in minutes 
    states[0].nextstate = 1;       // where do I go next?
    states[1].output = B00010101;
    states[1].wait = 1;
    states[1].nextstate = 2;
    states[2].output = B00100000;
    states[2].wait = 1;
    states[2].nextstate = 3;
    states[3].output = B00010000;
    states[3].wait = 1;
    states[3].nextstate = 4;
    states[4].output = B00001000;
    states[4].wait = 1;
    states[4].nextstate = 5;
    states[5].output = B00000100;
    states[5].wait = 1;
    states[5].nextstate = 6;
    states[6].output = B00000010;
    states[6].wait = 1;
    states[6].nextstate = 7;
    states[7].output = B00000001;
    states[7].wait = 1;
    states[7].nextstate = 0;
    
    for(i = 0; i < sizeof(statetable) * NUM_DEFAULT_STATES; i++)
    {
      EEPROM.write(2 + i, ((uint8_t *) states)[i]);
    }
  }
  else
  {
    Serial.println("Reading EEPROM");
    run_status = EEPROM.read(1);
    for(i = 0; i < sizeof(statetable) * NUM_STATES; i++)
      ((uint8_t *) states)[i] = EEPROM.read(2 + i);
  }
  
  //sprintf(line_buffer, "Run status=%d", run_status);
  //Serial.println(line_buffer);
  //for(i = 0; i < NUM_STATES; i++)
  // {
  //  sprintf(line_buffer, "** state %d -> output:%i wait:%d nextstate:%d", i,
  //           states[i].output, states[i].wait, states[i].nextstate);
  //  Serial.println(line_buffer);
  // }
  
  // set the initial state based on the state table
  wait_timer=states[0].wait;
  current_state=0;
  relay_status=states[current_state].output;
}


void loop(){
  uint16_t plen, dat_p;
  int8_t cmd, i;
  static uint16_t elapsed_minutes; // how many minutes have we been alive for?
  static unsigned long mscounter;       // a millisecond counter (1 minute = 60000ms)
  static uint16_t seconds;
 
  
  plen = es.ES_enc28j60PacketReceive(BUFFER_SIZE, buf);

  // do the state processing here, before we try to do the webpage processing...
  // 
 
  // heart of the timer - keep looking at the millisecond timer on the Arduino
  // and increment the a seconds and minutes counter
  
  //
  // only do the timer functions if the run status is on!
  if (run_status!=0) {
    if ( millis() - mscounter >999) {
        mscounter=millis();
        seconds++;
        
        // update the output status every second.
        //Serial.print("Setting output=");
        //Serial.println((int)states[current_state].output);
        relay_status= states[current_state].output;
        for (i=0; i<6; i++) {
          if ((lowByte(relay_status) & 1<<i)==0) digitalWrite(RELAY[i], LOW); else digitalWrite(RELAY[i], HIGH);
        }
        
        if (seconds==60) { 
            seconds=0; 
            elapsed_minutes++; 
            Serial.print("Elapsed minutes="); 
            Serial.println(elapsed_minutes);
        }
  
        //if the wait timer has reduced to zero, determine the next state
        if (--wait_timer==0) {
           current_state=states[current_state].nextstate;
           wait_timer=states[current_state].wait;
        }
   }
 }
 
 

        
        
        
  //plen will be unequal to zero if there is a valid packet (without crc error) 
  if(plen!=0){           
    // arp is broadcast if unknown but a host may also verify the mac address by sending it to a unicast address.
    if(es.ES_eth_type_is_arp_and_my_ip(buf,plen)){
       es.ES_make_arp_answer_from_request(buf);
       return;
    }

    // check if ip packets are for us:
    if(es.ES_eth_type_is_ip_and_my_ip(buf,plen)==0){
       return;
    }
    
    if(buf[IP_PROTO_P]==IP_PROTO_ICMP_V && buf[ICMP_TYPE_P]==ICMP_TYPE_ECHOREQUEST_V){
       es.ES_make_echo_reply_from_request(buf,plen);
       return;
    }
    
    // tcp port www start, compare only the lower byte
    if (buf[IP_PROTO_P]==IP_PROTO_TCP_V&&buf[TCP_DST_PORT_H_P]==0&&buf[TCP_DST_PORT_L_P]==mywwwport){
      if (buf[TCP_FLAGS_P] & TCP_FLAGS_SYN_V){
         es.ES_make_tcp_synack_from_syn(buf); // make_tcp_synack_from_syn does already send the syn,ack
         return;     
      }
      if (buf[TCP_FLAGS_P] & TCP_FLAGS_ACK_V){
        es.ES_init_len_info(buf); // init some data structures
        dat_p=es.ES_get_tcp_data_pointer();
        if (dat_p==0){ // we can possibly have no data, just ack:
          if (buf[TCP_FLAGS_P] & TCP_FLAGS_FIN_V){
            es.ES_make_tcp_ack_from_any(buf);
          }
          return;
        }
// Dont care about returning 200 messages to GET commands in this application
// if (strncmp("GET ",(char *)&(buf[dat_p]),4)!=0){
//          	// head, post and other methods for possible status codes see:
//            // http://www.w3.org/Protocols/rfc2616/rfc2616-sec10.html
//            plen=es.ES_fill_tcp_data_p(buf,0,PSTR("HTTP/1.0 200 OK\r\nContent-Type: text/html\r\n\r\n<h1>200 OK</h1>"));
//            goto SENDTCP; 
//        }


         if (strncmp("/ ",(char *)&(buf[dat_p+4]),2)==0){
                plen=print_manual_page(buf);
            goto SENDTCP;
         }
        
        cmd=get_key_val((char *)&(buf[dat_p+5]),"cmd");
         if (cmd==1){
             plen=print_program_page(buf,1);
        }
        if (cmd==2){
             plen=print_program_page(buf,2);
        }        
        if (cmd==3){
             plen=print_program_page(buf,3);
        }
        if (cmd==4){
             int tmr=get_key_val((char *)&(buf[dat_p+5]),"tmr"); // if there is a run status command - set that accordingly
             if (tmr==1) run_status=0;
             if (tmr==2) run_status=1;
             int rrly=get_key_val((char *)&(buf[dat_p+5]),"rly"); // if there is a selay set command - toggle the appropriate relay 
             if (rrly>=1)  relay_status^=1<<--rrly;  // toggle the state of the relay bit
             EEPROM.write(1, run_status);   

             for (i=0; i<6; i++) {
                    if ((lowByte(relay_status) & 1<<i)==0) digitalWrite(RELAY[i], LOW); else digitalWrite(RELAY[i], HIGH);
                  }
             plen=print_manual_page(buf);
        }
        if (cmd==5){
             uint8_t st=get_key_val((char *)&(buf[dat_p+5]),"st"); // if there is a state command - grab the state to change
             uint16_t wt=get_key_val((char *)&(buf[dat_p+5]),"wait"); // if there is a wait value, grab it 
             uint8_t nextst=get_key_val((char *)&(buf[dat_p+5]),"next"); // if there is a next value, grab it 
             if ((st!=-1) & (wt!=-1) & (nextst!=-1)) {
 
               //
               // update relay settings
               relay_status=0;
               if (find_key_val((char *)&(buf[dat_p+5]),"o1")) relay_status|=1;
               if (find_key_val((char *)&(buf[dat_p+5]),"o2")) relay_status|=2;
               if (find_key_val((char *)&(buf[dat_p+5]),"o3")) relay_status|=4;
               if (find_key_val((char *)&(buf[dat_p+5]),"o4")) relay_status|=8;
               if (find_key_val((char *)&(buf[dat_p+5]),"o5")) relay_status|=16;
               if (find_key_val((char *)&(buf[dat_p+5]),"o6")) relay_status|=32;
               //sprintf(line_buffer, "Doing state table change - state=%d, nextstate=%d, wait=%d output=%d", st, nextst, wt, relay_status);
               //Serial.println(line_buffer);

               states[st].nextstate = nextst; states[st].wait=wt; states[st].output=relay_status;
               
               for(i = 0; i < sizeof(statetable) * NUM_STATES; i++)
               {
                EEPROM.write(2 + i, ((uint8_t *) states)[i]);
               }
           
             }
             
             if (st<0 or st>NUM_STATES) st=0;
             plen=edit_program_page(buf,st);
         }        
        
SENDTCP:  es.ES_make_tcp_ack_from_any(buf); // send ack for http get
           es.ES_make_tcp_ack_with_data(buf,plen); // send data       
      }
    }
  }       
}



// The returned value is stored in the global var strbuf
uint8_t find_key_val(char *str,char *key)
{
        uint8_t found=0;
        uint8_t i=0;
        char *kp;
        kp=key;
        while(*str &&  *str!=' ' && found==0){
                if (*str == *kp){
                        kp++;
                        if (*kp == '\0'){
                                str++;
                                kp=key;
                                if (*str == '='){
                                        found=1;
                                }
                        }
                }else{
                        kp=key;
                }
                str++;
        }
        if (found==1){
                // copy the value to a buffer and terminate it with '\0'
                while(*str &&  *str!=' ' && *str!='&' && i<LINE_BUFFER_SIZE){
                        line_buffer[i]=*str;
                        i++;
                        str++;
                }
                line_buffer[i]='\0';
        }
        return(found);
}

int16_t get_key_val(char *str, char* cmd)
{
        r=-1;
        if (find_key_val(str,cmd)){
           if (*line_buffer < 0x3a && *line_buffer > 0x2f){
               // is a ASCII number, return it
               //r=(*line_buffer-0x30);
                r = atoi(line_buffer);
            }
        }
        return r;
}



uint16_t print_program_page(uint8_t *buf, uint8_t pagenum)
{

        uint8_t tempcount=0, count=0, rlycnt=0;
        uint8_t start=0, end=0;
        
        if (pagenum==1) {start=0; end=7;}
        if (pagenum==2) {start=7; end=14;}
        if (pagenum==3) {start=14; end=20;}
          
        uint16_t plen;

        //plen=es.ES_fill_tcp_data_p(buf,0,PSTR("HTTP/1.0 200 OK\r\nContent-Type: text/html\r\n\r\n"));
        tempcount=0;
        strcpy_P(line_buffer, (char*)pgm_read_word(&(string_table[0]))); // get header string 
        while (line_buffer[tempcount]) {
                   buf[TCP_CHECKSUM_L_P+3+plen]=line_buffer[tempcount++];
                   plen++;
                   }
        plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("<center><p><h1>Ethernet Relay Controller"));
        tempcount=0;
        strcpy_P(line_buffer, (char*)pgm_read_word(&(string_table[1]))); // get command string 
        while (line_buffer[tempcount]) {
                   buf[TCP_CHECKSUM_L_P+3+plen]=line_buffer[tempcount++];
                   plen++;
                   }
        plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("<h2>Current Program "));
 	plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("<h1>"));
 	plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("<table width=\"300\" border=1><tr><th>State<p>#<th>Wait<p>(seconds)<th>Next<p>State<th>Outputs<p>1 2 3 4 5 6</tr>"));

        //if ((start==0) & (end==0))
 	//   plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("<tr><td span=\"6\">Error invalid params passed to print_program_page()</tr>"));
        
        for (count=start; count<end; count++) {
          
              sprintf (relay_string, "123456");
              for (rlycnt=0; rlycnt<=6; rlycnt++) {
                if ((lowByte(states[count].output)  & 1<<rlycnt) ==0) relay_string[rlycnt]='-';
              }
              
              sprintf(line_buffer, "<tr align=\"center\"><td>%u<td>%u<td>%u<td>%c%c%c%c%c%c</tr>", 
                             count, states[count].wait, states[count].nextstate, relay_string[0],relay_string[1],relay_string[2],relay_string[3],relay_string[4],relay_string[5]);
                  // Serial.println(line_buffer);
              
              
              tempcount=0;
              while (line_buffer[tempcount]) {
                   buf[TCP_CHECKSUM_L_P+3+plen]=line_buffer[tempcount++];
                   plen++;
                   }
             
        }

 	plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("</table>") );

        plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("<h4><form method=GET action=/>State:<input type=text name=st>") );
        plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("<input type=hidden name=cmd value=5>"));     
        plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("<input type=\"submit\" value=\"Modify\" /></form>") );
       
        plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("</center><hr>"));
        tempcount=0;
        strcpy_P(line_buffer, (char*)pgm_read_word(&(string_table[2]))); // get copyright string 
        while (line_buffer[tempcount]) {
                   buf[TCP_CHECKSUM_L_P+3+plen]=line_buffer[tempcount++];
                   plen++;
                   }
        Serial.print("plen=");
        Serial.println(plen);
        
        return(plen);
}



//  /?cmd=4    Manual settings page - Make manual setting changes to relays and allow timer to be turned on or off
//  /?cmd=4?rly=1?tmr=0 - format for returning manual settings toggles relay1 state and turns off timer
uint16_t print_manual_page(uint8_t *buf)
{
        uint8_t tempcount=0, count=0;
       
        uint16_t plen;
       
        tempcount=0;
        strcpy_P(line_buffer, (char*)pgm_read_word(&(string_table[0]))); // get header string 
        while (line_buffer[tempcount]) {
                   buf[TCP_CHECKSUM_L_P+3+plen]=line_buffer[tempcount++];
                   plen++;
                   }
        plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("<center><p><h1>Ethernet Relay Controller"));
        tempcount=0;
        strcpy_P(line_buffer, (char*)pgm_read_word(&(string_table[1]))); // get command string 
        while (line_buffer[tempcount]) {
                   buf[TCP_CHECKSUM_L_P+3+plen]=line_buffer[tempcount++];
                   plen++;
                   }

        plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("<h2>Manual Control"));

        plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("<table border=\"1\" width=\"200\"><tr><th>Relay<th>Status<th>Toggle"));
        for (count=0; count<=5; count++) {
             if ((lowByte(relay_status) & (1<<count))==0)
                 {
                   sprintf(line_buffer, "<tr><td>%d<td>OFF<td><a href=/?cmd=4?rly=%d>Toggle</a>", count+1, count+1);
                 }
                 else
                 {
                   sprintf(line_buffer, "<tr><td>%d<td>ON<td><a href=/?cmd=4?rly=%d>Toggle</a>", count+1, count+1);
                 }
                
                tempcount=0;
                while (line_buffer[tempcount]) {
                   buf[TCP_CHECKSUM_L_P+3+plen]=line_buffer[tempcount++];
                   plen++;
                   }

        }

 	if (run_status==0) plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("<tr><th>Timer<td>OFF<td><a href=/?cmd=4?tmr=2>ON</a>") );
        else plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("<tr><th>Timer<td><a href=/?cmd=4?tmr=1>OFF</a><td>ON") );
  
	plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("</table><br>&nbsp;<p>  ") );
        plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("</center><hr>"));
        tempcount=0;
        strcpy_P(line_buffer, (char*)pgm_read_word(&(string_table[2]))); // get copyright string 
        while (line_buffer[tempcount]) {
                   buf[TCP_CHECKSUM_L_P+3+plen]=line_buffer[tempcount++];
                   plen++;
                   }
        Serial.print("plen=");
        Serial.println(plen);
        
  
        return(plen);
}


//  /?cmd=5?st=1   Edit Program page  - Allows changes to be made to the state settings
//  /?cmd=5?st=1?wait=2?next=2?o2=ON?o3=ON - format for returning state changes
uint16_t edit_program_page(uint8_t *buf, uint8_t prognum)
{
       
        uint8_t tempcount=0, count=0, rlycnt=0;
        uint8_t start=0, end=0;
        
        uint16_t plen;

        tempcount=0;
        strcpy_P(line_buffer, (char*)pgm_read_word(&(string_table[0]))); // get header string 
        while (line_buffer[tempcount]) {
                   buf[TCP_CHECKSUM_L_P+3+plen]=line_buffer[tempcount++];
                   plen++;
                   }

        plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("<center><p><h1>Ethernet Relay Controller</h1></p> "));
        tempcount=0;
        strcpy_P(line_buffer, (char*)pgm_read_word(&(string_table[1]))); // get command string 
        while (line_buffer[tempcount]) {
                   buf[TCP_CHECKSUM_L_P+3+plen]=line_buffer[tempcount++];
                   plen++;
                   }
        //plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("<h2> Modify Program </h2> "));
 	//plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("<h1>Current Settings:<br />"));
 	plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("<table border=1><tr><th>State<p>#<th>Wait<p>(seconds)<th>Next<p>State<th>Outputs<p>1 2 3 4 5 6</tr>"));

        sprintf (relay_string, "123456");
        for (rlycnt=0; rlycnt<=6; rlycnt++) {
             if ((lowByte(states[prognum].output) & 1<<rlycnt) ==0) relay_string[rlycnt]='-';
             }
              
        sprintf(line_buffer, "<tr align=center><td>%u<td>%u<td>%u<td>%c%c%c%c%c%c</tr>", 
                prognum, states[prognum].wait, states[prognum].nextstate, relay_string[0],relay_string[1],relay_string[2],relay_string[3],relay_string[4],relay_string[5]);
        // Serial.println(line_buffer);        
              
        tempcount=0;
        while (line_buffer[tempcount]) {
                buf[TCP_CHECKSUM_L_P+3+plen]=line_buffer[tempcount++];
                plen++;
                }

 	plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("</table>") );

        plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("<h4><form method=GET action=/>Wait Time:<input type=text name=wait>Next State:<input type=text name=next><br/>") );
        for (int relay=1; relay<=6; relay++)
        {
          sprintf(line_buffer, "Relay %d<input type=checkbox name=o%d>", relay,relay );
          tempcount=0;
          while (line_buffer[tempcount]) {
                buf[TCP_CHECKSUM_L_P+3+plen]=line_buffer[tempcount++];
                plen++;
                }
        }
        plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("<br />") );  
        plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("<input type=hidden name=cmd value=5>"));
        sprintf(line_buffer, "<input type=hidden name=st value=%d>", prognum );
        // Serial.println(line_buffer);        
              
        tempcount=0;
        while (line_buffer[tempcount]) {
                buf[TCP_CHECKSUM_L_P+3+plen]=line_buffer[tempcount++];
                plen++;
                }
     
        plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("<input type=\"submit\" value=\"Modify\" /></form>") );
         
        
        plen=es.ES_fill_tcp_data_p(buf,plen,PSTR("</center><hr>"));
        //tempcount=0;
        //strcpy_P(line_buffer, (char*)pgm_read_word(&(string_table[2]))); // get copyright string 
        //while (line_buffer[tempcount]) {
        //           buf[TCP_CHECKSUM_L_P+3+plen]=line_buffer[tempcount++];
        //           plen++;
        //           }
        Serial.print("plen=");
        Serial.println(plen);
        
        return(plen);
} 

